// SPDX-License-Identifier: BSD-3-Clause
/* Copyright (c) 2022, Linaro Ltd. */

#include <sys/mman.h>
#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "debugcc.h"

static struct gcc_mux gcc = {
	.mux = {
		.phys =	0x00100000,
		.size = 0x1f0000,

		.measure = measure_gcc,

		.enable_reg = 0x62004,
		.enable_mask = BIT(0),

		.mux_reg = 0x62024,
		.mux_mask = 0x1fff,

		.div_reg = 0x62000,
		.div_mask = 0xf,
		.div_val = 2,
	},

	.xo_div4_reg = 0x62008,
	.debug_ctl_reg = 0x62038,
	.debug_status_reg = 0x6203c,
};

static struct debug_mux cam_cc = {
	.phys = 0xad00000,
	.size = 0x20000,
	.block_name = "camcc",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x70,

	.enable_reg = 0xd008,
	.enable_mask = BIT(0),

	.mux_reg = 0xd100,
	.mux_mask = 0xff,

	.div_reg = 0xd004,
	.div_mask = 0x0f,
	.div_val = 2,
};

static struct debug_mux disp0_cc = {
	.phys = 0xaf00000,
	.size = 0x20000,
	.block_name = "disp0",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x79,

	.enable_reg = 0x500c,
	.enable_mask = BIT(0),

	.mux_reg = 0x7000,
	.mux_mask = 0xff,

	.div_reg = 0x5008,
	.div_mask = 0xf,
	.div_val = 4,
};

static struct debug_mux disp1_cc = {
	.phys = 0x22100000,
	.size = 0x20000,
	.block_name = "disp1",

	.measure = measure_leaf,
	.parent = &gcc.mux,
	.parent_mux_val = 0x82,

	.enable_reg = 0x500c,
	.enable_mask = BIT(0),

	.mux_reg = 0x7000,
	.mux_mask = 0xff,

	.div_reg = 0x5008,
	.div_mask = 0xf,
	.div_val = 4,
};

static struct debug_mux mc_cc = {
	.phys =	0x90ba000,
	.size = /* 0x54 */ 0x1000,
	.block_name = "mc",

	.measure = measure_mccc,
};

static struct measure_clk sc8280xp_clocks[] = {
	{ "gcc_aggre_noc_pcie0_tunnel_axi_clk",			&gcc.mux, 0x217 },
	{ "gcc_aggre_noc_pcie1_tunnel_axi_clk",			&gcc.mux, 0x218 },
	{ "gcc_aggre_noc_pcie_4_axi_clk",			&gcc.mux, 0x214 },
	{ "gcc_aggre_noc_pcie_south_sf_axi_clk",		&gcc.mux, 0x215 },
	{ "gcc_aggre_ufs_card_axi_clk",				&gcc.mux, 0x222 },
	{ "gcc_aggre_ufs_phy_axi_clk",				&gcc.mux, 0x221 },
	{ "gcc_aggre_usb3_mp_axi_clk",				&gcc.mux, 0x21b },
	{ "gcc_aggre_usb3_prim_axi_clk",			&gcc.mux, 0x219 },
	{ "gcc_aggre_usb3_sec_axi_clk",				&gcc.mux, 0x21a },
	{ "gcc_aggre_usb4_1_axi_clk",				&gcc.mux, 0x21d },
	{ "gcc_aggre_usb4_axi_clk",				&gcc.mux, 0x21c },
	{ "gcc_aggre_usb_noc_axi_clk",				&gcc.mux, 0x220 },
	{ "gcc_aggre_usb_noc_north_axi_clk",			&gcc.mux, 0x21f },
	{ "gcc_aggre_usb_noc_south_axi_clk",			&gcc.mux, 0x21e },
	{ "gcc_ahb2phy0_clk",					&gcc.mux, 0xfe },
	{ "gcc_ahb2phy2_clk",					&gcc.mux, 0xff },
	{ "gcc_camera_hf_axi_clk",				&gcc.mux, 0x6a },
	{ "gcc_camera_sf_axi_clk",				&gcc.mux, 0x6b },
	{ "gcc_camera_throttle_nrt_axi_clk",			&gcc.mux, 0x6d },
	{ "gcc_camera_throttle_rt_axi_clk",			&gcc.mux, 0x6c },
	{ "gcc_camera_throttle_xo_clk",				&gcc.mux, 0x6f },
	{ "gcc_cfg_noc_usb3_mp_axi_clk",			&gcc.mux, 0x2c },
	{ "gcc_cfg_noc_usb3_prim_axi_clk",			&gcc.mux, 0x2a },
	{ "gcc_cfg_noc_usb3_sec_axi_clk",			&gcc.mux, 0x2b },
	{ "gcc_cnoc_pcie0_tunnel_clk",				&gcc.mux, 0x1c },
	{ "gcc_cnoc_pcie1_tunnel_clk",				&gcc.mux, 0x1d },
	{ "gcc_cnoc_pcie4_qx_clk",				&gcc.mux, 0x18 },
	{ "gcc_ddrss_gpu_axi_clk",				&gcc.mux, 0x155 },
	{ "gcc_ddrss_pcie_sf_tbu_clk",				&gcc.mux, 0x156 },
	{ "gcc_disp1_hf_axi_clk",				&gcc.mux, 0x7d },
	{ "gcc_disp1_sf_axi_clk",				&gcc.mux, 0x7e },
	{ "gcc_disp1_throttle_nrt_axi_clk",			&gcc.mux, 0x80 },
	{ "gcc_disp1_throttle_rt_axi_clk",			&gcc.mux, 0x7f },
	{ "gcc_disp_hf_axi_clk",				&gcc.mux, 0x74 },
	{ "gcc_disp_sf_axi_clk",				&gcc.mux, 0x75 },
	{ "gcc_disp_throttle_nrt_axi_clk",			&gcc.mux, 0x77 },
	{ "gcc_disp_throttle_rt_axi_clk",			&gcc.mux, 0x76 },
	{ "gcc_emac0_axi_clk",					&gcc.mux, 0x246 },
	{ "gcc_emac0_ptp_clk",					&gcc.mux, 0x248 },
	{ "gcc_emac0_rgmii_clk",				&gcc.mux, 0x249 },
	{ "gcc_emac0_slv_ahb_clk",				&gcc.mux, 0x247 },
	{ "gcc_emac1_axi_clk",					&gcc.mux, 0x24a },
	{ "gcc_emac1_ptp_clk",					&gcc.mux, 0x24c },
	{ "gcc_emac1_rgmii_clk",				&gcc.mux, 0x24d },
	{ "gcc_emac1_slv_ahb_clk",				&gcc.mux, 0x24b },
	{ "gcc_gp1_clk",					&gcc.mux, 0x18c },
	{ "gcc_gp2_clk",					&gcc.mux, 0x18d },
	{ "gcc_gp3_clk",					&gcc.mux, 0x18e },
	{ "gcc_gp4_clk",					&gcc.mux, 0x290 },
	{ "gcc_gp5_clk",					&gcc.mux, 0x291 },
	{ "gcc_gpu_gpll0_clk_src",				&gcc.mux, 0x232 },
	{ "gcc_gpu_gpll0_div_clk_src",				&gcc.mux, 0x233 },
	{ "gcc_gpu_memnoc_gfx_clk",				&gcc.mux, 0x22e },
	{ "gcc_gpu_snoc_dvm_gfx_clk",				&gcc.mux, 0x231 },
	{ "gcc_gpu_tcu_throttle_ahb_clk",			&gcc.mux, 0x22b },
	{ "gcc_gpu_tcu_throttle_clk",				&gcc.mux, 0x22f },
	{ "gcc_pcie0_phy_rchng_clk",				&gcc.mux, 0x1d0 },
	{ "gcc_pcie1_phy_rchng_clk",				&gcc.mux, 0x19f },
	{ "gcc_pcie2a_phy_rchng_clk",				&gcc.mux, 0x1a9 },
	{ "gcc_pcie2b_phy_rchng_clk",				&gcc.mux, 0x1b3 },
	{ "gcc_pcie3a_phy_rchng_clk",				&gcc.mux, 0x1bd },
	{ "gcc_pcie3b_phy_rchng_clk",				&gcc.mux, 0x1c7 },
	{ "gcc_pcie4_phy_rchng_clk",				&gcc.mux, 0x196 },
	{ "gcc_pcie_0_aux_clk",					&gcc.mux, 0x1ce },
	{ "gcc_pcie_0_cfg_ahb_clk",				&gcc.mux, 0x1cd },
	{ "gcc_pcie_0_mstr_axi_clk",				&gcc.mux, 0x1cc },
	{ "gcc_pcie_0_pipe_clk",				&gcc.mux, 0x1cf },
	{ "gcc_pcie_0_slv_axi_clk",				&gcc.mux, 0x1cb },
	{ "gcc_pcie_0_slv_q2a_axi_clk",				&gcc.mux, 0x1ca },
	{ "gcc_pcie_1_aux_clk",					&gcc.mux, 0x19d },
	{ "gcc_pcie_1_cfg_ahb_clk",				&gcc.mux, 0x19c },
	{ "gcc_pcie_1_mstr_axi_clk",				&gcc.mux, 0x19b },
	{ "gcc_pcie_1_pipe_clk",				&gcc.mux, 0x19e },
	{ "gcc_pcie_1_slv_axi_clk",				&gcc.mux, 0x19a },
	{ "gcc_pcie_1_slv_q2a_axi_clk",				&gcc.mux, 0x199 },
	{ "gcc_pcie_2a_aux_clk",				&gcc.mux, 0x1a6 },
	{ "gcc_pcie_2a_cfg_ahb_clk",				&gcc.mux, 0x1a5 },
	{ "gcc_pcie_2a_mstr_axi_clk",				&gcc.mux, 0x1a4 },
	{ "gcc_pcie_2a_pipe_clk",				&gcc.mux, 0x1a7 },
	{ "gcc_pcie_2a_pipediv2_clk",				&gcc.mux, 0x1a8 },
	{ "gcc_pcie_2a_slv_axi_clk",				&gcc.mux, 0x1a3 },
	{ "gcc_pcie_2a_slv_q2a_axi_clk",			&gcc.mux, 0x1a2 },
	{ "gcc_pcie_2b_aux_clk",				&gcc.mux, 0x1b0 },
	{ "gcc_pcie_2b_cfg_ahb_clk",				&gcc.mux, 0x1af },
	{ "gcc_pcie_2b_mstr_axi_clk",				&gcc.mux, 0x1ae },
	{ "gcc_pcie_2b_pipe_clk",				&gcc.mux, 0x1b1 },
	{ "gcc_pcie_2b_pipediv2_clk",				&gcc.mux, 0x1b2 },
	{ "gcc_pcie_2b_slv_axi_clk",				&gcc.mux, 0x1ad },
	{ "gcc_pcie_2b_slv_q2a_axi_clk",			&gcc.mux, 0x1ac },
	{ "gcc_pcie_3a_aux_clk",				&gcc.mux, 0x1ba },
	{ "gcc_pcie_3a_cfg_ahb_clk",				&gcc.mux, 0x1b9 },
	{ "gcc_pcie_3a_mstr_axi_clk",				&gcc.mux, 0x1b8 },
	{ "gcc_pcie_3a_pipe_clk",				&gcc.mux, 0x1bb },
	{ "gcc_pcie_3a_pipediv2_clk",				&gcc.mux, 0x1bc },
	{ "gcc_pcie_3a_slv_axi_clk",				&gcc.mux, 0x1b7 },
	{ "gcc_pcie_3a_slv_q2a_axi_clk",			&gcc.mux, 0x1b6 },
	{ "gcc_pcie_3b_aux_clk",				&gcc.mux, 0x1c4 },
	{ "gcc_pcie_3b_cfg_ahb_clk",				&gcc.mux, 0x1c3 },
	{ "gcc_pcie_3b_mstr_axi_clk",				&gcc.mux, 0x1c2 },
	{ "gcc_pcie_3b_pipe_clk",				&gcc.mux, 0x1c5 },
	{ "gcc_pcie_3b_pipediv2_clk",				&gcc.mux, 0x1c6 },
	{ "gcc_pcie_3b_slv_axi_clk",				&gcc.mux, 0x1c1 },
	{ "gcc_pcie_3b_slv_q2a_axi_clk",			&gcc.mux, 0x1c0 },
	{ "gcc_pcie_4_aux_clk",					&gcc.mux, 0x193 },
	{ "gcc_pcie_4_cfg_ahb_clk",				&gcc.mux, 0x192 },
	{ "gcc_pcie_4_mstr_axi_clk",				&gcc.mux, 0x191 },
	{ "gcc_pcie_4_pipe_clk",				&gcc.mux, 0x194 },
	{ "gcc_pcie_4_pipediv2_clk",				&gcc.mux, 0x195 },
	{ "gcc_pcie_4_slv_axi_clk",				&gcc.mux, 0x190 },
	{ "gcc_pcie_4_slv_q2a_axi_clk",				&gcc.mux, 0x18f },
	{ "gcc_pcie_rscc_ahb_clk",				&gcc.mux, 0x8f },
	{ "gcc_pcie_rscc_xo_clk",				&gcc.mux, 0x8e },
	{ "gcc_pcie_throttle_cfg_clk",				&gcc.mux, 0x46 },
	{ "gcc_pdm2_clk",					&gcc.mux, 0x122 },
	{ "gcc_pdm_ahb_clk",					&gcc.mux, 0x120 },
	{ "gcc_pdm_xo4_clk",					&gcc.mux, 0x121 },
	{ "gcc_qmip_camera_nrt_ahb_clk",			&gcc.mux, 0x68 },
	{ "gcc_qmip_camera_rt_ahb_clk",				&gcc.mux, 0x69 },
	{ "gcc_qmip_disp1_ahb_clk",				&gcc.mux, 0x7b },
	{ "gcc_qmip_disp1_rot_ahb_clk",				&gcc.mux, 0x7c },
	{ "gcc_qmip_disp_ahb_clk",				&gcc.mux, 0x72 },
	{ "gcc_qmip_disp_rot_ahb_clk",				&gcc.mux, 0x73 },
	{ "gcc_qmip_video_cvp_ahb_clk",				&gcc.mux, 0x86 },
	{ "gcc_qmip_video_vcodec_ahb_clk",			&gcc.mux, 0x87 },
	{ "gcc_qupv3_wrap0_core_2x_clk",			&gcc.mux, 0x109 },
	{ "gcc_qupv3_wrap0_core_clk",				&gcc.mux, 0x108 },
	{ "gcc_qupv3_wrap0_qspi0_clk",				&gcc.mux, 0x112 },
	{ "gcc_qupv3_wrap0_s0_clk",				&gcc.mux, 0x10a },
	{ "gcc_qupv3_wrap0_s1_clk",				&gcc.mux, 0x10b },
	{ "gcc_qupv3_wrap0_s2_clk",				&gcc.mux, 0x10c },
	{ "gcc_qupv3_wrap0_s3_clk",				&gcc.mux, 0x10d },
	{ "gcc_qupv3_wrap0_s4_clk",				&gcc.mux, 0x10e },
	{ "gcc_qupv3_wrap0_s5_clk",				&gcc.mux, 0x10f },
	{ "gcc_qupv3_wrap0_s6_clk",				&gcc.mux, 0x110 },
	{ "gcc_qupv3_wrap0_s7_clk",				&gcc.mux, 0x111 },
	{ "gcc_qupv3_wrap1_core_2x_clk",			&gcc.mux, 0x116 },
	{ "gcc_qupv3_wrap1_core_clk",				&gcc.mux, 0x115 },
	{ "gcc_qupv3_wrap1_qspi0_clk",				&gcc.mux, 0x11f },
	{ "gcc_qupv3_wrap1_s0_clk",				&gcc.mux, 0x117 },
	{ "gcc_qupv3_wrap1_s1_clk",				&gcc.mux, 0x118 },
	{ "gcc_qupv3_wrap1_s2_clk",				&gcc.mux, 0x119 },
	{ "gcc_qupv3_wrap1_s3_clk",				&gcc.mux, 0x11a },
	{ "measure_only_gcc_qupv3_wrap1_s4_clk",		&gcc.mux, 0x11b },
	{ "gcc_qupv3_wrap1_s5_clk",				&gcc.mux, 0x11c },
	{ "gcc_qupv3_wrap1_s6_clk",				&gcc.mux, 0x11d },
	{ "gcc_qupv3_wrap1_s7_clk",				&gcc.mux, 0x11e },
	{ "gcc_qupv3_wrap2_core_2x_clk",			&gcc.mux, 0x251 },
	{ "gcc_qupv3_wrap2_core_clk",				&gcc.mux, 0x250 },
	{ "gcc_qupv3_wrap2_qspi0_clk",				&gcc.mux, 0x25a },
	{ "gcc_qupv3_wrap2_s0_clk",				&gcc.mux, 0x252 },
	{ "gcc_qupv3_wrap2_s1_clk",				&gcc.mux, 0x253 },
	{ "gcc_qupv3_wrap2_s2_clk",				&gcc.mux, 0x254 },
	{ "gcc_qupv3_wrap2_s3_clk",				&gcc.mux, 0x255 },
	{ "gcc_qupv3_wrap2_s4_clk",				&gcc.mux, 0x256 },
	{ "gcc_qupv3_wrap2_s5_clk",				&gcc.mux, 0x257 },
	{ "gcc_qupv3_wrap2_s6_clk",				&gcc.mux, 0x258 },
	{ "gcc_qupv3_wrap2_s7_clk",				&gcc.mux, 0x259 },
	{ "gcc_qupv3_wrap_0_m_ahb_clk",				&gcc.mux, 0x106 },
	{ "gcc_qupv3_wrap_0_s_ahb_clk",				&gcc.mux, 0x107 },
	{ "measure_only_gcc_qupv3_wrap_1_m_ahb_clk",		&gcc.mux, 0x113 },
	{ "measure_only_gcc_qupv3_wrap_1_s_ahb_clk",		&gcc.mux, 0x114 },
	{ "gcc_qupv3_wrap_2_m_ahb_clk",				&gcc.mux, 0x24e },
	{ "gcc_qupv3_wrap_2_s_ahb_clk",				&gcc.mux, 0x24f },
	{ "gcc_sdcc2_ahb_clk",					&gcc.mux, 0x101 },
	{ "gcc_sdcc2_apps_clk",					&gcc.mux, 0x100 },
	{ "gcc_sdcc4_ahb_clk",					&gcc.mux, 0x104 },
	{ "gcc_sdcc4_apps_clk",					&gcc.mux, 0x103 },
	{ "gcc_sys_noc_usb_axi_clk",				&gcc.mux, 0x14 },
	{ "gcc_ufs_card_ahb_clk",				&gcc.mux, 0x1d4 },
	{ "gcc_ufs_card_axi_clk",				&gcc.mux, 0x1d3 },
	{ "gcc_ufs_card_ice_core_clk",				&gcc.mux, 0x1da },
	{ "gcc_ufs_card_phy_aux_clk",				&gcc.mux, 0x1db },
	{ "gcc_ufs_card_rx_symbol_0_clk",			&gcc.mux, 0x1d6 },
	{ "gcc_ufs_card_rx_symbol_1_clk",			&gcc.mux, 0x1dc },
	{ "gcc_ufs_card_tx_symbol_0_clk",			&gcc.mux, 0x1d5 },
	{ "gcc_ufs_card_unipro_core_clk",			&gcc.mux, 0x1d9 },
	{ "gcc_ufs_phy_ahb_clk",				&gcc.mux, 0x1e0 },
	{ "gcc_ufs_phy_axi_clk",				&gcc.mux, 0x1df },
	{ "gcc_ufs_phy_ice_core_clk",				&gcc.mux, 0x1e6 },
	{ "gcc_ufs_phy_phy_aux_clk",				&gcc.mux, 0x1e7 },
	{ "gcc_ufs_phy_rx_symbol_0_clk",			&gcc.mux, 0x1e2 },
	{ "gcc_ufs_phy_rx_symbol_1_clk",			&gcc.mux, 0x1e8 },
	{ "gcc_ufs_phy_tx_symbol_0_clk",			&gcc.mux, 0x1e1 },
	{ "gcc_ufs_phy_unipro_core_clk",			&gcc.mux, 0x1e5 },
	{ "gcc_usb30_mp_master_clk",				&gcc.mux, 0xc5 },
	{ "gcc_usb30_mp_mock_utmi_clk",				&gcc.mux, 0xc7 },
	{ "gcc_usb30_prim_master_clk",				&gcc.mux, 0xb6 },
	{ "gcc_usb30_prim_mock_utmi_clk",			&gcc.mux, 0xb8 },
	{ "gcc_usb30_sec_master_clk",				&gcc.mux, 0xbf },
	{ "gcc_usb30_sec_mock_utmi_clk",			&gcc.mux, 0xc1 },
	{ "gcc_usb3_mp_phy_aux_clk",				&gcc.mux, 0xc8 },
	{ "gcc_usb3_mp_phy_com_aux_clk",			&gcc.mux, 0xc9 },
	{ "gcc_usb3_mp_phy_pipe_0_clk",				&gcc.mux, 0xca },
	{ "gcc_usb3_mp_phy_pipe_1_clk",				&gcc.mux, 0xcb },
	{ "gcc_usb3_prim_phy_aux_clk",				&gcc.mux, 0xb9 },
	{ "gcc_usb3_prim_phy_com_aux_clk",			&gcc.mux, 0xba },
	{ "gcc_usb3_prim_phy_pipe_clk",				&gcc.mux, 0xbb },
	{ "gcc_usb3_sec_phy_aux_clk",				&gcc.mux, 0xc2 },
	{ "gcc_usb3_sec_phy_com_aux_clk",			&gcc.mux, 0xc3 },
	{ "gcc_usb3_sec_phy_pipe_clk",				&gcc.mux, 0xc4 },
	{ "gcc_usb4_1_cfg_ahb_clk",				&gcc.mux, 0xf3 },
	{ "gcc_usb4_1_dp_clk",					&gcc.mux, 0xf0 },
	{ "gcc_usb4_1_master_clk",				&gcc.mux, 0xec },
	{ "gcc_usb4_1_phy_p2rr2p_pipe_clk",			&gcc.mux, 0xf8 },
	{ "gcc_usb4_1_phy_pcie_pipe_clk",			&gcc.mux, 0xee },
	{ "gcc_usb4_1_phy_rx0_clk",				&gcc.mux, 0xf4 },
	{ "gcc_usb4_1_phy_rx1_clk",				&gcc.mux, 0xf5 },
	{ "gcc_usb4_1_phy_usb_pipe_clk",			&gcc.mux, 0xf2 },
	{ "gcc_usb4_1_sb_if_clk",				&gcc.mux, 0xed },
	{ "gcc_usb4_1_sys_clk",					&gcc.mux, 0xef },
	{ "gcc_usb4_1_tmu_clk",					&gcc.mux, 0xf1 },
	{ "gcc_usb4_cfg_ahb_clk",				&gcc.mux, 0xe1 },
	{ "gcc_usb4_dp_clk",					&gcc.mux, 0xde },
	{ "gcc_usb4_master_clk",				&gcc.mux, 0xda },
	{ "gcc_usb4_phy_p2rr2p_pipe_clk",			&gcc.mux, 0xe6 },
	{ "gcc_usb4_phy_pcie_pipe_clk",				&gcc.mux, 0xdc },
	{ "gcc_usb4_phy_rx0_clk",				&gcc.mux, 0xe2 },
	{ "gcc_usb4_phy_rx1_clk",				&gcc.mux, 0xe3 },
	{ "gcc_usb4_phy_usb_pipe_clk",				&gcc.mux, 0xe0 },
	{ "gcc_usb4_sb_if_clk",					&gcc.mux, 0xdb },
	{ "gcc_usb4_sys_clk",					&gcc.mux, 0xdd },
	{ "gcc_usb4_tmu_clk",					&gcc.mux, 0xdf },
	{ "gcc_video_axi0_clk",					&gcc.mux, 0x88 },
	{ "gcc_video_axi1_clk",					&gcc.mux, 0x89 },
	{ "gcc_video_cvp_throttle_clk",				&gcc.mux, 0x8b },
	{ "gcc_video_vcodec_throttle_clk",			&gcc.mux, 0x8a },
	{ "measure_only_cnoc_clk",				&gcc.mux, 0x22 },
	{ "measure_only_gcc_camera_ahb_clk",			&gcc.mux, 0x67 },
	{ "measure_only_gcc_camera_xo_clk",			&gcc.mux, 0x6e },
	{ "measure_only_gcc_disp1_ahb_clk",			&gcc.mux, 0x7a },
	{ "measure_only_gcc_disp1_xo_clk",			&gcc.mux, 0x81 },
	{ "measure_only_gcc_disp_ahb_clk",			&gcc.mux, 0x71 },
	{ "measure_only_gcc_disp_xo_clk",			&gcc.mux, 0x78 },
	{ "measure_only_gcc_gpu_cfg_ahb_clk",			&gcc.mux, 0x22a },
	{ "measure_only_gcc_video_ahb_clk",			&gcc.mux, 0x85 },
	{ "measure_only_gcc_video_xo_clk",			&gcc.mux, 0x8c },
	{ "measure_only_ipa_2x_clk",				&gcc.mux, 0x225 },
	{ "measure_only_memnoc_clk",				&gcc.mux, 0x15b },
	{ "measure_only_snoc_clk",				&gcc.mux, 0xc },
	{ "disp0_cc_mdss_ahb1_clk",				&disp0_cc, 0x39 },
	{ "disp0_cc_mdss_ahb_clk",				&disp0_cc, 0x38 },
	{ "disp0_cc_mdss_byte0_clk",				&disp0_cc, 0x19 },
	{ "disp0_cc_mdss_byte0_intf_clk",			&disp0_cc, 0x1a },
	{ "disp0_cc_mdss_byte1_clk",				&disp0_cc, 0x1b },
	{ "disp0_cc_mdss_byte1_intf_clk",			&disp0_cc, 0x1c },
	{ "disp0_cc_mdss_dptx0_aux_clk",			&disp0_cc, 0x23 },
	{ "disp0_cc_mdss_dptx0_link_clk",			&disp0_cc, 0x1f },
	{ "disp0_cc_mdss_dptx0_link_intf_clk",			&disp0_cc, 0x20 },
	{ "disp0_cc_mdss_dptx0_pixel0_clk",			&disp0_cc, 0x24 },
	{ "disp0_cc_mdss_dptx0_pixel1_clk",			&disp0_cc, 0x25 },
	{ "disp0_cc_mdss_dptx0_usb_router_link_intf_clk",	&disp0_cc, 0x21 },
	{ "disp0_cc_mdss_dptx1_aux_clk",			&disp0_cc, 0x31 },
	{ "disp0_cc_mdss_dptx1_link_clk",			&disp0_cc, 0x2a },
	{ "disp0_cc_mdss_dptx1_link_intf_clk",			&disp0_cc, 0x2b },
	{ "disp0_cc_mdss_dptx1_pixel0_clk",			&disp0_cc, 0x26 },
	{ "disp0_cc_mdss_dptx1_pixel1_clk",			&disp0_cc, 0x27 },
	{ "disp0_cc_mdss_dptx1_usb_router_link_intf_clk",	&disp0_cc, 0x2c },
	{ "disp0_cc_mdss_dptx2_aux_clk",			&disp0_cc, 0x32 },
	{ "disp0_cc_mdss_dptx2_link_clk",			&disp0_cc, 0x2d },
	{ "disp0_cc_mdss_dptx2_link_intf_clk",			&disp0_cc, 0x2e },
	{ "disp0_cc_mdss_dptx2_pixel0_clk",			&disp0_cc, 0x28 },
	{ "disp0_cc_mdss_dptx2_pixel1_clk",			&disp0_cc, 0x29 },
	{ "disp0_cc_mdss_dptx3_aux_clk",			&disp0_cc, 0x37 },
	{ "disp0_cc_mdss_dptx3_link_clk",			&disp0_cc, 0x34 },
	{ "disp0_cc_mdss_dptx3_link_intf_clk",			&disp0_cc, 0x35 },
	{ "disp0_cc_mdss_dptx3_pixel0_clk",			&disp0_cc, 0x33 },
	{ "disp0_cc_mdss_esc0_clk",				&disp0_cc, 0x1d },
	{ "disp0_cc_mdss_esc1_clk",				&disp0_cc, 0x1e },
	{ "disp0_cc_mdss_mdp1_clk",				&disp0_cc, 0x12 },
	{ "disp0_cc_mdss_mdp_clk",				&disp0_cc, 0x11 },
	{ "disp0_cc_mdss_mdp_lut1_clk",				&disp0_cc, 0x16 },
	{ "disp0_cc_mdss_mdp_lut_clk",				&disp0_cc, 0x15 },
	{ "disp0_cc_mdss_non_gdsc_ahb_clk",			&disp0_cc, 0x3a },
	{ "disp0_cc_mdss_pclk0_clk",				&disp0_cc, 0xf },
	{ "disp0_cc_mdss_pclk1_clk",				&disp0_cc, 0x10 },
	{ "disp0_cc_mdss_rot1_clk",				&disp0_cc, 0x14 },
	{ "disp0_cc_mdss_rot_clk",				&disp0_cc, 0x13 },
	{ "disp0_cc_mdss_rscc_ahb_clk",				&disp0_cc, 0x3c },
	{ "disp0_cc_mdss_rscc_vsync_clk",			&disp0_cc, 0x3b },
	{ "disp0_cc_mdss_vsync1_clk",				&disp0_cc, 0x18 },
	{ "disp0_cc_mdss_vsync_clk",				&disp0_cc, 0x17 },
	{ "disp0_cc_sleep_clk",					&disp0_cc, 0x46 },
	{ "disp0_cc_xo_clk",					&disp0_cc, 0x45 },
	{ "disp1_cc_mdss_ahb1_clk",				&disp1_cc, 0x39 },
	{ "disp1_cc_mdss_ahb_clk",				&disp1_cc, 0x38 },
	{ "disp1_cc_mdss_byte0_clk",				&disp1_cc, 0x19 },
	{ "disp1_cc_mdss_byte0_intf_clk",			&disp1_cc, 0x1a },
	{ "disp1_cc_mdss_byte1_clk",				&disp1_cc, 0x1b },
	{ "disp1_cc_mdss_byte1_intf_clk",			&disp1_cc, 0x1c },
	{ "disp1_cc_mdss_dptx0_aux_clk",			&disp1_cc, 0x23 },
	{ "disp1_cc_mdss_dptx0_link_clk",			&disp1_cc, 0x1f },
	{ "disp1_cc_mdss_dptx0_link_intf_clk",			&disp1_cc, 0x20 },
	{ "disp1_cc_mdss_dptx0_pixel0_clk",			&disp1_cc, 0x24 },
	{ "disp1_cc_mdss_dptx0_pixel1_clk",			&disp1_cc, 0x25 },
	{ "disp1_cc_mdss_dptx0_usb_router_link_intf_clk",	&disp1_cc, 0x21 },
	{ "disp1_cc_mdss_dptx1_aux_clk",			&disp1_cc, 0x31 },
	{ "disp1_cc_mdss_dptx1_link_clk",			&disp1_cc, 0x2a },
	{ "disp1_cc_mdss_dptx1_link_intf_clk",			&disp1_cc, 0x2b },
	{ "disp1_cc_mdss_dptx1_pixel0_clk",			&disp1_cc, 0x26 },
	{ "disp1_cc_mdss_dptx1_pixel1_clk",			&disp1_cc, 0x27 },
	{ "disp1_cc_mdss_dptx1_usb_router_link_intf_clk",	&disp1_cc, 0x2c },
	{ "disp1_cc_mdss_dptx2_aux_clk",			&disp1_cc, 0x32 },
	{ "disp1_cc_mdss_dptx2_link_clk",			&disp1_cc, 0x2d },
	{ "disp1_cc_mdss_dptx2_link_intf_clk",			&disp1_cc, 0x2e },
	{ "disp1_cc_mdss_dptx2_pixel0_clk",			&disp1_cc, 0x28 },
	{ "disp1_cc_mdss_dptx2_pixel1_clk",			&disp1_cc, 0x29 },
	{ "disp1_cc_mdss_dptx3_aux_clk",			&disp1_cc, 0x37 },
	{ "disp1_cc_mdss_dptx3_link_clk",			&disp1_cc, 0x34 },
	{ "disp1_cc_mdss_dptx3_link_intf_clk",			&disp1_cc, 0x35 },
	{ "disp1_cc_mdss_dptx3_pixel0_clk",			&disp1_cc, 0x33 },
	{ "disp1_cc_mdss_esc0_clk",				&disp1_cc, 0x1d },
	{ "disp1_cc_mdss_esc1_clk",				&disp1_cc, 0x1e },
	{ "disp1_cc_mdss_mdp1_clk",				&disp1_cc, 0x12 },
	{ "disp1_cc_mdss_mdp_clk",				&disp1_cc, 0x11 },
	{ "disp1_cc_mdss_mdp_lut1_clk",				&disp1_cc, 0x16 },
	{ "disp1_cc_mdss_mdp_lut_clk",				&disp1_cc, 0x15 },
	{ "disp1_cc_mdss_non_gdsc_ahb_clk",			&disp1_cc, 0x3a },
	{ "disp1_cc_mdss_pclk0_clk",				&disp1_cc, 0xf },
	{ "disp1_cc_mdss_pclk1_clk",				&disp1_cc, 0x10 },
	{ "disp1_cc_mdss_rot1_clk",				&disp1_cc, 0x14 },
	{ "disp1_cc_mdss_rot_clk",				&disp1_cc, 0x13 },
	{ "disp1_cc_mdss_rscc_ahb_clk",				&disp1_cc, 0x3c },
	{ "disp1_cc_mdss_rscc_vsync_clk",			&disp1_cc, 0x3b },
	{ "disp1_cc_mdss_vsync1_clk",				&disp1_cc, 0x18 },
	{ "disp1_cc_mdss_vsync_clk",				&disp1_cc, 0x17 },
	{ "disp1_cc_sleep_clk",					&disp1_cc, 0x46 },
	{ "disp1_cc_xo_clk",					&disp1_cc, 0x45 },
	{ "measure_only_mccc_clk", 				&mc_cc, 0x50 },
	{ "cam_cc_mclk0_clk",					&cam_cc, 0x1 },
	{ "cam_cc_mclk1_clk",					&cam_cc, 0x2 },
	{ "cam_cc_mclk2_clk",					&cam_cc, 0x3 },
	{ "cam_cc_mclk3_clk",					&cam_cc, 0x4 },
	{ "cam_cc_csi0phytimer_clk",				&cam_cc, 0x5 },
	{ "cam_cc_csiphy0_clk",					&cam_cc, 0x6},
	{ "cam_cc_csi1phytimer_clk",				&cam_cc, 0x7},
	{ "cam_cc_csiphy1_clk",					&cam_cc, 0x8},
	{ "cam_cc_csi2phytimer_clk",				&cam_cc, 0x9},
	{ "cam_cc_csiphy2_clk",					&cam_cc, 0xa},
	{ "cam_cc_bps_clk",					&cam_cc, 0xb},
	{ "cam_cc_bps_axi_clk",					&cam_cc, 0xc},
	{ "cam_cc_bps_areg_clk",				&cam_cc, 0xd},
	{ "cam_cc_bps_ahb_clk",					&cam_cc, 0xe},
	{ "cam_cc_ipe_0_clk",					&cam_cc, 0xf},
	{ "cam_cc_ipe_0_axi_clk",				&cam_cc, 0x10},
	{ "cam_cc_ipe_0_areg_clk",				&cam_cc, 0x11},
	{ "cam_cc_ipe_0_ahb_clk",				&cam_cc, 0x12},
	{ "cam_cc_ipe_1_clk",					&cam_cc, 0x13},
	{ "cam_cc_ipe_1_axi_clk",				&cam_cc, 0x14},
	{ "cam_cc_ipe_1_areg_clk",				&cam_cc, 0x15},
	{ "cam_cc_ipe_1_ahb_clk",				&cam_cc, 0x16},
	{ "cam_cc_ife_0_clk",					&cam_cc, 0x17},
	{ "cam_cc_ife_0_dsp_clk",				&cam_cc, 0x18},
	{ "cam_cc_ife_0_csid_clk",				&cam_cc, 0x19},
	{ "cam_cc_ife_0_cphy_rx_clk",				&cam_cc, 0x1a},
	{ "cam_cc_ife_0_axi_clk",				&cam_cc, 0x1b},
	{ "cam_cc_spdm_ife_1_clk",				&cam_cc, 0x1c},
	{ "cam_cc_ife_1_clk",					&cam_cc, 0x1d},
	{ "cam_cc_ife_1_dsp_clk",				&cam_cc, 0x1e},
	{ "cam_cc_ife_1_csid_clk",				&cam_cc, 0x1f},
	{ "cam_cc_ife_1_cphy_rx_clk",				&cam_cc, 0x20},
	{ "cam_cc_ife_1_axi_clk",				&cam_cc, 0x21},
	{ "cam_cc_ife_lite_0_clk",				&cam_cc, 0x22},
	{ "cam_cc_ife_lite_0_csid_clk",				&cam_cc, 0x23},
	{ "cam_cc_ife_lite_0_cphy_rx_clk",			&cam_cc, 0x24},
	{ "cam_cc_jpeg_clk",					&cam_cc, 0x25},
	{ "cam_cc_icp_clk",					&cam_cc, 0x26},
	{ "cam_cc_camnoc_axi_clk",				&cam_cc, 0x27},
	{ "cam_cc_spdm_ife_1_csid_clk",				&cam_cc, 0x28},
	{ "cam_cc_pll_lock_monitor_clk",			&cam_cc, 0x29},
	{ "cam_cc_cci_0_clk",					&cam_cc, 0x2a},
	{ "cam_cc_lrme_clk",					&cam_cc, 0x2b},
	{ "cam_cc_cpas_ahb_clk",				&cam_cc, 0x2c},
	{ "cam_cc_spdm_bps_clk",				&cam_cc, 0x2d},
	{ "cam_cc_core_ahb_clk",				&cam_cc, 0x2e},
	{ "cam_cc_spdm_ipe_0_clk",				&cam_cc, 0x2f},
	{ "cam_cc_spdm_ipe_1_clk",				&cam_cc, 0x30},
	{ "cam_cc_spdm_ife_0_clk",				&cam_cc, 0x31},
	{ "cam_cc_spdm_ife_0_csid_clk",				&cam_cc, 0x32},
	{ "cam_cc_camnoc_dcd_xo_clk",				&cam_cc, 0x33},
	{ "cam_cc_spdm_jpeg_clk",				&cam_cc, 0x34},
	{ "cam_cc_csi3phytimer_clk",				&cam_cc, 0x35},
	{ "cam_cc_csiphy3_clk",					&cam_cc, 0x36},
	{ "cam_cc_icp_ahb_clk",					&cam_cc, 0x37},
	{ "cam_cc_ife_lite_1_clk",				&cam_cc, 0x38},
	{ "cam_cc_ife_lite_1_csid_clk",				&cam_cc, 0x39},
	{ "cam_cc_ife_lite_1_cphy_rx_clk",			&cam_cc, 0x3a},
	{ "cam_cc_cci_1_clk",					&cam_cc, 0x3b},
	{ "cam_cc_gdsc_clk",					&cam_cc, 0x3c},
	{ "cam_cc_qdss_debug_clk",				&cam_cc, 0x3d},
	{ "cam_cc_qdss_debug_xo_clk",				&cam_cc, 0x3e},
	{ "cam_cc_sleep_clk",					&cam_cc, 0x3f},
	{ "csiphy0_cam_cc_debug_clk",				&cam_cc, 0x40},
	{ "csiphy1_cam_cc_debug_clk",				&cam_cc, 0x41},
	{ "csiphy2_cam_cc_debug_clk",				&cam_cc, 0x42},
	{ "csiphy3_cam_cc_debug_clk",				&cam_cc, 0x43},
	{ "cam_cc_ife_2_clk",					&cam_cc, 0x44},
	{ "cam_cc_spdm_ife_2_clk",				&cam_cc, 0x45},
	{ "cam_cc_ife_2_dsp_clk",				&cam_cc, 0x46},
	{ "cam_cc_ife_2_csid_clk",				&cam_cc, 0x47},
	{ "cam_cc_spdm_ife_2_csid_clk",				&cam_cc, 0x48},
	{ "cam_cc_ife_2_cphy_rx_clk",				&cam_cc, 0x49},
	{ "cam_cc_ife_2_axi_clk",				&cam_cc, 0x4a},
	{ "cam_cc_ife_3_clk",					&cam_cc, 0x4b},
	{ "cam_cc_spdm_ife_3_clk",				&cam_cc, 0x4c},
	{ "cam_cc_ife_3_dsp_clk",				&cam_cc, 0x4d},
	{ "cam_cc_ife_3_csid_clk",				&cam_cc, 0x4e},
	{ "cam_cc_spdm_ife_3_csid_clk",				&cam_cc, 0x4f},
	{ "cam_cc_ife_3_cphy_rx_clk",				&cam_cc, 0x50},
	{ "cam_cc_ife_3_axi_clk",				&cam_cc, 0x51},
	{ "cam_cc_ife_lite_2_clk",				&cam_cc, 0x55},
	{ "cam_cc_ife_lite_2_csid_clk",				&cam_cc, 0x56},
	{ "cam_cc_ife_lite_2_cphy_rx_clk",			&cam_cc, 0x57},
	{ "cam_cc_ife_lite_3_clk",				&cam_cc, 0x58},
	{ "cam_cc_ife_lite_3_csid_clk",				&cam_cc, 0x59},
	{ "cam_cc_ife_lite_3_cphy_rx_clk",			&cam_cc, 0x5a},
	{ "cam_cc_cci_2_clk",					&cam_cc, 0x5b},
	{ "cam_cc_cci_3_clk",					&cam_cc, 0x5c},
	{ "cam_cc_mclk4_clk",					&cam_cc, 0x5d},
	{ "cam_cc_mclk5_clk",					&cam_cc, 0x5e},
	{ "cam_cc_mclk6_clk",					&cam_cc, 0x5f},
	{ "cam_cc_mclk7_clk",					&cam_cc, 0x60},
	{}
};

struct debugcc_platform sc8280xp_debugcc = {
	"sc8280xp",
	sc8280xp_clocks,
};
