# Copyright 2014-2021 SUSE LLC
# SPDX-License-Identifier: GPL-2.0-or-later

use Test::Most;

use FindBin;
use lib "$FindBin::Bin/lib", "$FindBin::Bin/../external/os-autoinst-common/lib";

use Test::Warnings ':report_warnings';
use Test::Output 'combined_like';
use Test::MockModule;
use Mojolicious;
use Mojo::Base -signatures;
use Mojo::Log;
use OpenQA::App;
use OpenQA::Config;
use OpenQA::Constants qw(DEFAULT_WORKER_TIMEOUT MAX_TIMER);
use OpenQA::Test::TimeLimit '4';
use OpenQA::Setup;
use OpenQA::JobGroupDefaults;
use OpenQA::Task::Job::Limit;
use Mojo::File 'tempdir';
use Time::Seconds;

my $quiet_log = Mojo::Log->new(level => 'warn');

sub read_config {
    my ($app, $msg) = @_;
    $msg //= 'reading config from default';
    local $Test::Builder::Level = $Test::Builder::Level + 1;
    combined_like sub { OpenQA::Setup::read_config($app) }, qr/fallback to default/, $msg;
    return $app->config;
}

subtest 'Test configuration default modes' => sub {
    # test with a completely empty config file to check defaults
    # note: We cannot use no config file at all because then the lookup would fallback to a system configuration.
    my $t_dir = tempdir;
    $t_dir->child('openqa.ini')->touch;
    local $ENV{OPENQA_CONFIG} = $t_dir;

    OpenQA::App->set_singleton(my $app = Mojolicious->new(log => $quiet_log));
    $app->mode('test');
    my $config = read_config($app, 'reading config from default with mode test');
    is(length($config->{_openid_secret}), 16, 'config has openid_secret');
    my $test_config = {
        global => {
            appname => 'openQA',
            branding => 'openSUSE',
            hsts => 365,
            audit_enabled => 1,
            max_rss_limit => 0,
            profiling_enabled => 0,
            monitoring_enabled => 0,
            mcp_enabled => 'no',
            hide_asset_types => 'repo',
            file_security_policy => 'download-prompt',
            recognized_referers => [],
            changelog_file => '/usr/share/openqa/public/Changelog',
            job_investigate_ignore => '"(JOBTOKEN|NAME)"',
            job_investigate_git_timeout => 20,
            job_investigate_git_log_limit => 200,
            search_results_limit => 50000,
            worker_timeout => DEFAULT_WORKER_TIMEOUT,
            force_result_regex => '',
            parallel_children_collapsable_results_sel => ' .status:not(.result_passed):not(.result_softfailed)',
            auto_clone_limit => 20,
            api_hmac_time_tolerance => 300,
            frontpage_builds => 3,
        },
        rate_limits => {
            search => 5,
        },
        auth => {
            method => 'Fake',
            require_for_assets => 0,
        },
        'scm git' => {
            update_remote => '',
            update_branch => '',
            do_push => 'no',
            do_cleanup => 'no',
            git_auto_clone => 'yes',
            git_auto_commit => '',
            git_auto_update => 'no',
            git_auto_update_method => 'best-effort',
            checkout_needles_sha => 'no',
            allow_arbitrary_url_fetch => 'no',
            temp_needle_refs_retention => 2 * ONE_MINUTE,
        },
        'scheduler' => {
            max_job_scheduled_time => 7,
            max_running_jobs => -1,
        },
        openid => {
            provider => 'https://www.opensuse.org/openid/user/',
            httpsonly => 1,
        },
        oauth2 => {
            provider => '',
            key => '',
            secret => '',
            authorize_url => '',
            token_url => '',
            user_url => '',
            token_scope => '',
            token_label => '',
            id_from => 'id',
            nickname_from => '',
            unique_name => '',
        },
        hypnotoad => {
            listen => ['http://localhost:9526/'],
            proxy => 1,
        },
        audit => {
            blacklist => '',
            blocklist => '',
        },
        plugin_links => {
            operator => {},
            admin => {}
        },
        amqp => {
            reconnect_timeout => 5,
            publish_attempts => 10,
            publish_retry_delay => 1,
            publish_retry_delay_factor => 1.75,
            url => 'amqp://guest:guest@localhost:5672/',
            exchange => 'pubsub',
            topic_prefix => 'suse',
            cacertfile => '',
            certfile => '',
            keyfile => ''
        },
        obs_rsync => {
            home => '',
            retry_interval => 60,
            retry_max_count => 1400,
            queue_limit => 200,
            concurrency => 2,
            project_status_url => '',
            username => '',
            ssh_key_file => '',
        },
        cleanup => {
            concurrent => 0,
        },
        default_group_limits => {
            asset_size_limit => OpenQA::JobGroupDefaults::SIZE_LIMIT_GB,
            log_storage_duration => OpenQA::JobGroupDefaults::KEEP_LOGS_IN_DAYS,
            important_log_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_LOGS_IN_DAYS,
            result_storage_duration => OpenQA::JobGroupDefaults::KEEP_RESULTS_IN_DAYS,
            important_result_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_RESULTS_IN_DAYS,
            job_storage_duration => OpenQA::JobGroupDefaults::KEEP_JOBS_IN_DAYS,
            important_job_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_JOBS_IN_DAYS,
        },
        no_group_limits => {
            log_storage_duration => OpenQA::JobGroupDefaults::KEEP_LOGS_IN_DAYS,
            important_log_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_LOGS_IN_DAYS,
            result_storage_duration => OpenQA::JobGroupDefaults::KEEP_RESULTS_IN_DAYS,
            important_result_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_RESULTS_IN_DAYS,
            job_storage_duration => OpenQA::JobGroupDefaults::KEEP_JOBS_IN_DAYS,
            important_job_storage_duration => OpenQA::JobGroupDefaults::KEEP_IMPORTANT_JOBS_IN_DAYS,
        },
        minion_task_triggers => {
            on_job_done => [],
        },
        misc_limits => {
            untracked_assets_storage_duration => 14,
            result_cleanup_max_free_percentage => 100,
            asset_cleanup_max_free_percentage => 100,
            screenshot_cleanup_batch_size => OpenQA::Task::Job::Limit::DEFAULT_SCREENSHOTS_PER_BATCH,
            screenshot_cleanup_batches_per_minion_job => OpenQA::Task::Job::Limit::DEFAULT_BATCHES_PER_MINION_JOB,
            minion_job_max_age => ONE_WEEK,
            generic_default_limit => 10000,
            generic_max_limit => 100000,
            tests_overview_max_jobs => 2000,
            all_tests_default_finished_jobs => 500,
            all_tests_max_finished_jobs => 5000,
            list_templates_default_limit => 5000,
            list_templates_max_limit => 20000,
            next_jobs_default_limit => 500,
            next_jobs_max_limit => 10000,
            previous_jobs_default_limit => 500,
            previous_jobs_max_limit => 10000,
            job_settings_max_recent_jobs => 20000,
            assets_default_limit => 100000,
            assets_max_limit => 200000,
            max_online_workers => 1000,
            wait_for_grutask_retries => 6,
            worker_limit_retry_delay => ONE_HOUR / 4,
            mcp_max_result_size => 500000,
            max_job_time_prio_scale => 100,
            scheduled_product_min_storage_duration => 34,
        },
        archiving => {
            archive_preserved_important_jobs => 0,
        },
        job_settings_ui => {
            keys_to_render_as_links => '',
            default_data_dir => 'data',
        },
        influxdb => {
            ignored_failed_minion_jobs => '',
        },
        carry_over => {
            lookup_depth => 10,
            state_changes_limit => 3,
        },
        secrets => {github_token => ''},
    };

    # Test configuration generation with "test" mode
    $test_config->{_openid_secret} = $config->{_openid_secret};
    $test_config->{logging}->{level} = 'debug';
    $test_config->{global}->{service_port_delta} = 2;
    is ref delete $config->{global}->{auto_clone_regex}, 'Regexp', 'auto_clone_regex parsed as regex';
    ok delete $config->{'test_preset example'}, 'default values for example tests assigned';
    is_deeply $config, $test_config, '"test" configuration';

    # Test configuration generation with "development" mode
    $app = Mojolicious->new(mode => 'development');
    $config = read_config($app, 'reading config from default with mode development');
    $test_config->{_openid_secret} = $config->{_openid_secret};
    $test_config->{global}->{service_port_delta} = 2;
    delete $config->{global}->{auto_clone_regex};
    delete $config->{'test_preset example'};
    is_deeply $config, $test_config, 'right "development" configuration';

    # Test configuration generation with an unknown mode (should fallback to default)
    $app = Mojolicious->new(mode => 'foo_bar');
    $config = read_config($app, 'reading config from default with mode foo_bar');
    $test_config->{_openid_secret} = $config->{_openid_secret};
    $test_config->{auth}->{method} = 'OpenID';
    $test_config->{global}->{service_port_delta} = 2;
    delete $config->{global}->{auto_clone_regex};
    delete $config->{'test_preset example'};
    delete $test_config->{logging};
    is_deeply $config, $test_config, 'right default configuration';
};

subtest 'Test configuration override from file' => sub {
    my $t_dir = tempdir;
    local $ENV{OPENQA_CONFIG} = $t_dir;
    OpenQA::App->set_singleton(my $app = Mojolicious->new(log => $quiet_log));
    my @data = (
        "[global]\n",
        "suse_mirror=http://blah/\n",
        "recognized_referers = bugzilla.suse.com bugzilla.opensuse.org progress.opensuse.org github.com\n",
        "[audit]\n",
        "blacklist = job_grab job_done\n",
        "[assets/storage_duration]\n",
        "-CURRENT = 40\n",
        "[minion_task_triggers]\n",
        "on_job_done = spam eggs\n",
        "[default_group_limits]\n",
        "result_storage_duration = 0\n",
        "[no_group_limits]\n",
        "result_storage_duration = 731\n",
        "[influxdb]\n",
        "ignored_failed_minion_jobs = foo boo\n"

    );
    $t_dir->child('openqa.ini')->spew(join '', @data);
    combined_like sub { OpenQA::Setup::read_config($app) }, qr/Deprecated.*blacklist/, 'notice about deprecated key';

    ok -e $t_dir->child('openqa.ini');
    ok $app->config->{global}->{suse_mirror} eq 'http://blah/', 'suse mirror';
    ok $app->config->{audit}->{blocklist} eq 'job_grab job_done', 'audit blocklist migrated from deprecated key name';
    is $app->config->{'assets/storage_duration'}->{'-CURRENT'}, 40, 'assets/storage_duration';

    is_deeply(
        $app->config->{global}->{recognized_referers},
        [qw(bugzilla.suse.com bugzilla.opensuse.org progress.opensuse.org github.com)],
        'referers parsed correctly'
    );

    is_deeply($app->config->{minion_task_triggers}->{on_job_done},
        [qw(spam eggs)], 'parse minion task triggers correctly');
    is_deeply($app->config->{influxdb}->{ignored_failed_minion_jobs},
        [qw(foo boo)], 'parse ignored_failed_minion_jobs correctly');

    is $app->config->{default_group_limits}->{job_storage_duration}, 0,
      'default job_storage_duration extended to result_storage_duration';
    is $app->config->{no_group_limits}->{job_storage_duration}, 731,
      'default job_storage_duration extended to result_storage_duration (no group)';
};

subtest 'trim whitespace characters from both ends of openqa.ini value' => sub {
    my $t_dir = tempdir;
    local $ENV{OPENQA_CONFIG} = $t_dir;
    OpenQA::App->set_singleton(my $app = Mojolicious->new(log => $quiet_log));
    my $data = '
        [global]
        appname =  openQA  
        hide_asset_types = repo iso  
        recognized_referers =   bugzilla.suse.com   progress.opensuse.org github.com
    ';
    $t_dir->child('openqa.ini')->spew($data);
    my $global_config = OpenQA::Setup::read_config($app)->{global};
    is $global_config->{appname}, 'openQA', 'appname';
    is $global_config->{hide_asset_types}, 'repo iso', 'hide_asset_types';
    is_deeply $global_config->{recognized_referers},
      [qw(bugzilla.suse.com progress.opensuse.org github.com)],
      'recognized_referers';
};

subtest 'Validation of worker timeout' => sub {
    my $app = Mojolicious->new(config => {global => {worker_timeout => undef}}, log => $quiet_log);
    my $configured_timeout = \$app->config->{global}->{worker_timeout};
    OpenQA::App->set_singleton($app);
    subtest 'too low worker_timeout' => sub {
        $$configured_timeout = MAX_TIMER - 1;
        combined_like { OpenQA::Setup::_validate_worker_timeout($app) } qr/worker_timeout.*invalid/, 'warning logged';
        is $$configured_timeout, DEFAULT_WORKER_TIMEOUT, 'rejected';
    };
    subtest 'minimum worker_timeout' => sub {
        $$configured_timeout = MAX_TIMER;
        OpenQA::Setup::_validate_worker_timeout($app);
        is $$configured_timeout, MAX_TIMER, 'accepted';
    };
    subtest 'invalid worker_timeout' => sub {
        $$configured_timeout = 'invalid';
        combined_like { OpenQA::Setup::_validate_worker_timeout($app) } qr/worker_timeout.*invalid/, 'warning logged';
        is $$configured_timeout, DEFAULT_WORKER_TIMEOUT, 'rejected';
    };
};

subtest 'Validation of file_security_policy' => sub {
    my %config;
    my $app = Mojolicious->new(config => \%config, log => $quiet_log);
    for my $value (qw(insecure-browsing download-prompt)) {
        $config{file_security_policy} = $value;
        OpenQA::Setup::_validate_security_policy($app, \%config);
        is $config{file_security_policy}, $value, "$value is valid";
    }
    $config{file_security_policy} = 'wrong';
    combined_like { OpenQA::Setup::_validate_security_policy($app, \%config) } qr/Invalid.*security/, 'warning logged';
    is $config{file_security_policy}, 'download-prompt', 'default to "download-prompt" on invalid value';
    is $config{file_domain}, undef, 'file_domain not populated yet';
    $config{file_security_policy} = 'domain:openqa-foo';
    OpenQA::Setup::_validate_security_policy($app, \%config);
    is $config{file_domain}, 'openqa-foo', 'file_domain populated via "domain:"';
};

subtest 'Multiple config files' => sub {
    my $t_dir = tempdir;
    my $openqa_d = $t_dir->child('openqa.ini.d')->make_path;
    local $ENV{OPENQA_CONFIG} = $t_dir;
    OpenQA::App->set_singleton(my $app = Mojolicious->new(log => $quiet_log));
    my $data_main = "[global]\nappname =  openQA main config\nhide_asset_types = repo iso\n";
    my $data_01 = "[global]\nappname =  openQA override 1\nbranding = fedora";
    my $data_02 = "[global]\nappname =  openQA override 2";
    $t_dir->child('openqa.ini')->spew($data_main);
    $openqa_d->child('01-appname-and-scm.ini')->spew($data_01);
    $openqa_d->child('02-appname.ini')->spew($data_02);
    my $global_config = OpenQA::Setup::read_config($app)->{global};
    is $global_config->{appname}, 'openQA override 2', 'appname overriden by config from openqa.ini.d, last one wins';
    is $global_config->{branding}, 'fedora', 'scm set by config from openqa.ini.d, not overriden';
    is $global_config->{hide_asset_types}, 'repo iso', 'types set from main config, not overriden';
};

subtest 'Lookup precedence/hiding' => sub {
    my $t_dir = tempdir;
    my @args = (undef, 'openqa.ini');
    my $config_mock = Test::MockModule->new('OpenQA::Config');
    $config_mock->redefine(_config_dirs => [["$t_dir/override"], ["$t_dir/home"], ["$t_dir/admin", "$t_dir/package"]]);

    my @expected;
    is_deeply lookup_config_files(@args), \@expected, 'no config files found';

    @expected = ("$t_dir/package/openqa.ini", "$t_dir/package/openqa.ini.d/packager-drop-in.ini");
    $t_dir->child('package')->make_path->child('openqa.ini')->touch->sibling('openqa.ini.d')
      ->make_path->child('packager-drop-in.ini')->touch;
    is_deeply lookup_config_files(@args), \@expected, 'found config from package';

    splice @expected, 0, 0, "$t_dir/admin/openqa.ini.d/admin-drop-in.ini";
    $t_dir->child('admin')->make_path->child('openqa.ini.d')->make_path->child('admin-drop-in.ini')->touch;
    is_deeply lookup_config_files(@args), \@expected, 'additional config from admin does not hide config from packager';

    @expected = ("$t_dir/admin/openqa.ini", "$t_dir/admin/openqa.ini.d/admin-drop-in.ini");
    $t_dir->child('admin')->child('openqa.ini')->touch;
    is_deeply lookup_config_files(@args), \@expected, 'main config from admin hides config from packager';

    @expected = ("$t_dir/home/openqa.ini.d/home-drop-in.ini");
    $t_dir->child('home')->child('openqa.ini.d')->make_path->child('home-drop-in.ini')->touch;
    is_deeply lookup_config_files(@args), \@expected, 'drop-in in home hides all other config';

    @expected = ("$t_dir/override/openqa.ini.d/override-drop-in.ini");
    $t_dir->child('override')->child('openqa.ini.d')->make_path->child('override-drop-in.ini')->touch;
    is_deeply lookup_config_files(@args), \@expected, 'drop-in in overriden dir hides all other config';
};

done_testing();
