# -*- coding: utf-8 -*- #
# frozen_string_literal: true

require 'open-uri'

LUA_SYNTAX_VERSION = "5.3"
LUA_SYNTAX_URI = "https://www.lua.org/manual/#{LUA_SYNTAX_VERSION}/"
LUA_KEYWORDS_FILE = "./lib/rouge/lexers/lua/keywords.rb"

namespace :builtins do
  task :lua do
    generator = Rouge::Tasks::Builtins::Lua.new

    input    = URI.open(LUA_SYNTAX_URI) { |f| f.read }
    keywords = generator.extract_keywords(input)

    output = generator.render_output(keywords)

    File.write(LUA_KEYWORDS_FILE, output)
  end
end

module Rouge
  module Tasks
    module Builtins
      class Lua
        def extract_keywords(input)
          keywords = Hash.new { |h,k| h[k] = Array.new }

          input.scrub.scan(%r[<a href="manual.html#pdf-(.+?)">\1</a>]im) do |m|
            name = m[0].downcase

            key = if name.start_with? "package"
                    "modules"
                  elsif name.include? "."
                    name.split(".").first
                  elsif %w(require module).include? name
                    "modules"
                  else
                    "basic"
                  end

            keywords[key].push name
          end

          keywords
        end

        def render_output(keywords, &b)
          return enum_for(:render_output, keywords).to_a.join("\n") unless b

          yield   "# -*- coding: utf-8 -*- #"
          yield   "# frozen_string_literal: true"
          yield   ""
          yield   "# DO NOT EDIT"
          yield   ""
          yield   "# This file is automatically generated by `rake builtins:lua`."
          yield   "# See tasks/builtins/lua.rake for more info."
          yield   ""
          yield   "module Rouge"
          yield   "  module Lexers"
          yield   "    class Lua"
          yield   "      def self.builtins"
          yield   "        @builtins ||= {}.tap do |b|"
          keywords.each do |n, fs|
            yield "          b[#{n.inspect}] = Set.new #{fs.inspect}"
          end
          yield   "        end"
          yield   "      end"
          yield   "    end"
          yield   "  end"
          yield   "end"
        end
      end
    end
  end
end
